/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Collections;
using System.Reflection;

using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.Handles;
using Borland.Eco.DesignTimeAttributes;
using Borland.Eco.Globalization;

namespace Borland.Eco.Persistence
{
	public interface IRemotePersistenceMapperProvider
	{
		IPersistenceMapper GetPersistenceMapper(int typeSystemHash);
	}


	[ToolboxItem(false)]
	[Designer("Borland.Eco.Persistence.Design.PersistenceMapperProviderDesigner, Borland.Eco.Persistence.Design", typeof(IRootDesigner))]
	[ComponentHelpAttribute("Borland.Eco.Persistence.Design.PersistenceMapperProviderInspector", "Borland.Eco.Persistence.Design")]
	public class PersistenceMapperProvider: Component, IRemotePersistenceMapperProvider, IPersistencemapperProvider // intended to be overridden, must not be abstract
	{
		private System.Type m_EcoSpaceType;
		private PersistenceMapperDb m_PersistenceMapper;
		private IPersistenceMapper m_MarchalByRefMapper;
		static Hashtable C_Instances = new Hashtable();

		///<exception cref="ArgumentNullException">Thrown if <paramref name="mapperType"/> is null</exception>
		///<exception cref="ArgumentException">Thrown if <paramref name="mapperType"/> is not a subclass of <see cref="PersistenceMapperProvider"/>.</exception>
		public static PersistenceMapperProvider GetInstance(Type mapperType)
		{
			if (mapperType == null) throw new ArgumentNullException("mapperType"); // do not localize
			if (!mapperType.IsSubclassOf(typeof(PersistenceMapperProvider)))
				throw new ArgumentException(PersistenceStringRes.sMapperTypeNotInHierarchy, "mapperType"); // do not localize
			PersistenceMapperProvider pmp = null;
			lock (C_Instances.SyncRoot)
			{
				pmp = (PersistenceMapperProvider)C_Instances[mapperType];
				if (pmp == null)
				{
					pmp = (PersistenceMapperProvider)Activator.CreateInstance(mapperType);
					C_Instances[mapperType] = pmp;
				}
			}
			return pmp;
		}

		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryEco")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyPersistenceMapper")]
		[DefaultValue(null)]
		public PersistenceMapperDb PersistenceMapper
		{
			get { return m_PersistenceMapper; }
			set
			{
				CheckUnused();
				m_PersistenceMapper = value;
			}
		}

		private void CheckUnused()
		{
			if (m_MarchalByRefMapper != null)
				throw new InvalidOperationException(PersistenceStringRes.sPMapperInUse);
		}

		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryEco")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyEcoSpaceType")]
		[TypeConverter("Borland.Eco.Handles.Design.EcoSpaceTypeTypeConverter, Borland.Eco.Handles.Design")]
		[DefaultValue(null)]
		public System.Type EcoSpaceType
		{
			get { return  m_EcoSpaceType; }
			set
			{
				CheckUnused();
				m_EcoSpaceType = value;
			}
		}

		///<exception cref="InvalidOperationException">Thrown if <paramref name="required"/> is true and <see cref="EcoSpaceType"/> not set.</exception>
		///<exception cref="InvalidOperationException">Thrown if the method GetTypeSystemService is not defined.</exception>
		///<exception cref="InvalidOperationException">Thrown if the return value from GetTypeSystemService is not an <see cref="IEcoServiceProvider"/>.</exception>
		public ITypeSystemService GetTypeSystemService(bool required)
		{
			if (EcoSpaceType == null)
			{
				if (required)
					throw new InvalidOperationException(PersistenceStringRes.sEcoSpaceTypeNotSet);
				else
					return null;
			}
			MethodInfo mi = EcoSpaceType.GetMethod(DefaultEcoSpace.GetTypeSystemServiceName, BindingFlags.Static | BindingFlags.Public);
			if (mi == null)
				throw new InvalidOperationException(PersistenceStringRes.sGetTypeSystemServiceMissing(this.ToString()));
			ITypeSystemService tss = mi.Invoke(null, null) as ITypeSystemService;
			if (tss == null)
				throw new InvalidOperationException(PersistenceStringRes.sNotATypeSystemProvider(this.ToString()));
			return tss;
		}

		///<exception cref="InvalidOperationException">Thrown if <see cref="PersistenceMapper"/> is null.</exception>
		IPersistenceMapper IRemotePersistenceMapperProvider.GetPersistenceMapper(int typeSystemHash)
		{
			if (PersistenceMapper == null)
				throw new InvalidOperationException(PersistenceStringRes.sPersistenceMapperNotSet);
			// FIXME check hashcode
			if (m_MarchalByRefMapper == null)
			{
				m_MarchalByRefMapper = PersistenceMapper.GetPersistenceMapper(GetTypeSystemService(true));
			}
			return m_MarchalByRefMapper;
		}

		///<exception cref="InvalidOperationException">Thrown if <see cref="PersistenceMapper"/> is null.</exception>
		///<exception cref="ArgumentException">Thrown if the provider is in use with another <see cref="ITypeSystemService"/>.</exception>
		public IPersistenceMapper GetPersistenceMapper(ITypeSystemService typeSystemService)
		{
			if (PersistenceMapper == null)
				throw new InvalidOperationException(PersistenceStringRes.sPersistenceMapperNotSet);
			if (typeSystemService != GetTypeSystemService(true))
				throw new ArgumentException(PersistenceStringRes.sProviderUsedWithOtherTypeSystem);
			return PersistenceMapper.GetPersistenceMapper(typeSystemService);
		}

		///<exception cref="InvalidOperationException">Thrown if <see cref="PersistenceMapper"/> is null.</exception>
		public void ReturnPersistenceMapper(IPersistenceMapper persistenceMapper)
		{
			if (PersistenceMapper == null)
				throw new InvalidOperationException(PersistenceStringRes.sPersistenceMapperNotSet);
			PersistenceMapper.ReturnPersistenceMapper(persistenceMapper);
		}
	}

}
